/****************************************************************************
** libebml : parse EBML files, see http://embl.sourceforge.net/
**
** <file/class description>
**
** Copyright (C) 2002-2003 Steve Lhomme.  All rights reserved.
**
** This file is part of libmatroska.
**
** This file may be distributed under the terms of the Q Public License
** as defined by Trolltech AS of Norway and appearing in the file
** LICENSE.QPL included in the packaging of this file.
**
** This file may be distributed and/or modified under the terms of the
** GNU General Public License version 2 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.
**
** Licensees holding an other license may use this file in accordance with 
** the Agreement provided with the Software.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
** See http://www.matroska.org/license/qpl/ for QPL licensing information.
** See http://www.matroska.org/license/gpl/ for GPL licensing information.
**
** Contact license@matroska.org if any conditions of this licensing are
** not clear to you.
**
**********************************************************************/

/*!
	\file
	\version \$Id: EbmlUnicodeString.h,v 1.4 2003/08/01 03:41:59 jcsston Exp $
	\author Steve Lhomme     <robux4 @ users.sf.net>
	\author Moritz Bunkus <moritz @ bunkus.org>
	\author Jory Stone       <jcsston @ toughguy.net>
*/
#ifndef LIBEBML_UNICODE_STRING_H
#define LIBEBML_UNICODE_STRING_H

//#define UTFSTRING_EXTENDED	

#include <string>

#ifdef UTFSTRING_EXTENDED
#include <stdarg.h>
#include "ebml/Debug.h"
#endif // UTFSTRING_EXTENDED

#include "EbmlConfig.h"
#include "EbmlTypes.h"
#include "EbmlElement.h"

START_LIBEBML_NAMESPACE

/*!
  \class UTFstring
  A class storing strings in a wchar_t (ie, in UCS-2 or UCS-4)
  \note inspired by wstring which is not available everywhere
*/
class UTFstring {
public:
	typedef wchar_t value_type;

	UTFstring();
	UTFstring(const wchar_t *); // should be NULL terminated
	UTFstring(const UTFstring &);
#ifdef UTFSTRING_EXTENDED	
	UTFstring(const char *); 
	void clear() {*this = (wchar_t)0;};
	/// Append string to end of current string
	void append(const wchar_t *value);
	void append(const char *value);
	void append(const wchar_t value);
	void append(const char value);
	

	UTFstring Left(unsigned long count);
	UTFstring Right(unsigned long count);
	UTFstring Mid(unsigned long start, unsigned long count);

	/// Get the count that a char appears in the string
	size_t Count(wchar_t value);
	
	//int64 Find(wchar_t *value);
	int64 Find(wchar_t value, bool bBackwards = false);

  /// Get all characters before the first occurance of ch
  /// (returns the whole string if ch not found)
  UTFstring BeforeFirst(wchar_t ch);
  /// Get all characters before the last occurence of ch
  /// (returns empty string if ch not found)
  UTFstring BeforeLast(wchar_t ch);
  /// Get all characters after the first occurence of ch
  /// (returns empty string if ch not found)
  UTFstring AfterFirst(wchar_t ch);
  /// Get all characters after the last occurence of ch
	/// (returns the whole string if ch not found)
  UTFstring AfterLast(wchar_t ch);

	/// printf style Formating functions
	static UTFstring Format(const wchar_t *value, ...);
	static UTFstring Format(const char *value, ...);

#endif // UTFSTRING_EXTENDED

	virtual ~UTFstring();
	bool operator==(const UTFstring&) const;	
	UTFstring & operator=(const UTFstring &);
	UTFstring & operator=(const wchar_t *);
	UTFstring & operator=(wchar_t);
#ifdef UTFSTRING_EXTENDED	
	UTFstring & operator=(const char *);	
	UTFstring & operator=(char);

	UTFstring & operator+=(const wchar_t *value) {append(value); return *this;};
	UTFstring & operator+=(wchar_t value) {append(value); return *this;};
	UTFstring & operator+=(const char *value) {append(value); return *this;};
	UTFstring & operator+=(char value) {append(value); return *this;};

	UTFstring & operator<<(const wchar_t *value) {append(value); return *this;};
	UTFstring & operator<<(wchar_t value) {append(value); return *this;};
	UTFstring & operator<<(const char *value) {append(value); return *this;};
	UTFstring & operator<<(char value) {append(value); return *this;};

	UTFstring operator+(const wchar_t *value) {append(value); return *this;};
	UTFstring operator+(const char *value) {append(value); return *this;};
	UTFstring operator+(wchar_t value) {append(value); return *this;};
	UTFstring operator+(char value) {append(value); return *this;};

	wchar_t operator[](size_t index) const {assert(index < _Length); return _Data[index];}
	char operator[](size_t index) {assert(index < _Length); _UpdateASCII(); return _DataASCII[index];}
#endif // UTFSTRING_EXTENDED	
	/// Return length of string
	size_t length() const {return _Length;}

	operator const wchar_t*() const {return _Data;}
	const wchar_t* c_str() const {return _Data;}
#ifdef UTFSTRING_EXTENDED
	//operator const char*() {EBML_TRACE("Warning: UTFstring being converted to char* without explict .mb_str() function."); return mb_str(); }
	const char* mb_str() {_UpdateASCII(); return _DataASCII;}
#endif // UTFSTRING_EXTENDED	

	const std::string & GetUTF8() const {return UTF8string;}
	void SetUTF8(const std::string &);

protected:
#ifdef UTFSTRING_EXTENDED
	/// Checks if the ASCII copy of the data needs to be updated
	void _UpdateASCII();

	/// Flag true if the ASCII version needs to be updated
	bool m_bUpdateASCII;
	/// ASCII version of _Data
	char *_DataASCII; 
#endif // UTFSTRING_EXTENDED

	size_t _Length; ///< length of the UCS string excluding the \0
	wchar_t* _Data; ///< internal UCS representation	
	std::string UTF8string;
	static bool wcscmp(const wchar_t *str1, const wchar_t *str2);
	void UpdateFromUTF8();
	void UpdateFromUCS2();
};


/*!
    \class EbmlUnicodeString
    \brief Handle all operations on a Unicode string EBML element
	\note internally treated as a string made of wide characters (ie UCS-2 or UCS-4 depending on the machine)
*/
class EbmlUnicodeString : public EbmlElement {
	public:
		EbmlUnicodeString();
		EbmlUnicodeString(const UTFstring & DefaultValue);
		EbmlUnicodeString(const EbmlElement & ElementToClone);
		EbmlUnicodeString(const EbmlElement & ElementToClone, const UTFstring & aDefaultValue);
	
		virtual ~EbmlUnicodeString() {}
	
		bool ValidateSize() const {return true;} // any size is possible
		uint32 RenderData(IOCallback & output, bool bForceRender, bool bSaveDefault = false);
		uint64 ReadData(IOCallback & input);
		uint64 UpdateSize(bool bSaveDefault = false, bool bForceRender = false);
	
		EbmlUnicodeString & operator=(const UTFstring &); ///< platform dependant code
		operator const UTFstring &() const {return Value;}
	
    void SetDefaultValue(UTFstring & aValue) {assert(!DefaultIsSet); DefaultValue = aValue; DefaultIsSet = true;}
    
		UTFstring DefaultVal() const {assert(DefaultIsSet); return DefaultValue;}

		bool IsDefaultValue() const {
			return (DefaultISset() && Value == DefaultValue);
		}

	protected:
		UTFstring Value; /// The actual value of the element
		UTFstring DefaultValue;
};

END_LIBEBML_NAMESPACE

#endif // LIBEBML_UNICODE_STRING_H
