/*****************************************************************************
 * open.cpp : wxWindows plugin for vlc
 *****************************************************************************
 * Copyright (C) 2000-2001 VideoLAN
 * $Id: open.cpp,v 1.35 2003/07/26 14:19:04 gbazin Exp $
 *
 * Authors: Gildas Bazin <gbazin@netcourrier.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************/

/*****************************************************************************
 * Preamble
 *****************************************************************************/
#include <stdlib.h>                                      /* malloc(), free() */
#include <errno.h>                                                 /* ENOMEM */
#include <string.h>                                            /* strerror() */
#include <stdio.h>

#include <vlc/vlc.h>

#ifdef WIN32                                                 /* mingw32 hack */
#undef Yield
#undef CreateDialog
#endif

/* Let vlc take care of the i18n stuff */
#define WXINTL_NO_GETTEXT_MACRO

#include <wx/wxprec.h>
#include <wx/wx.h>
#include <wx/notebook.h>
#include <wx/textctrl.h>
#include <wx/combobox.h>
#include <wx/spinctrl.h>
#include <wx/statline.h>
#include <wx/tokenzr.h>

#include <vlc/intf.h>

#include "wxwindows.h"

#ifndef wxRB_SINGLE
#   define wxRB_SINGLE 0
#endif

/*****************************************************************************
 * Event Table.
 *****************************************************************************/

/* IDs for the controls and the menu commands */
enum
{
    Notebook_Event = wxID_HIGHEST,
    MRL_Event,

    FileBrowse_Event,
    FileName_Event,

    DiscType_Event,
    DiscDevice_Event,
    DiscTitle_Event,
    DiscChapter_Event,

    NetType_Event,
    NetRadio1_Event, NetRadio2_Event, NetRadio3_Event,
    NetPort1_Event, NetPort2_Event, NetPort3_Event,
    NetAddr1_Event, NetAddr2_Event, NetAddr3_Event,

    SubsFileEnable_Event,
    SubsFileSettings_Event,

    SoutEnable_Event,
    SoutSettings_Event,

    DemuxDump_Event,
    DemuxDumpEnable_Event,
    DemuxDumpBrowse_Event,
};

BEGIN_EVENT_TABLE(OpenDialog, wxFrame)
    /* Button events */
    EVT_BUTTON(wxID_OK, OpenDialog::OnOk)
    EVT_BUTTON(wxID_CANCEL, OpenDialog::OnCancel)

    EVT_NOTEBOOK_PAGE_CHANGED(Notebook_Event, OpenDialog::OnPageChange)

    EVT_TEXT(MRL_Event, OpenDialog::OnMRLChange) 

    /* Events generated by the file panel */
    EVT_TEXT(FileName_Event, OpenDialog::OnFilePanelChange)
    EVT_BUTTON(FileBrowse_Event, OpenDialog::OnFileBrowse)

    /* Events generated by the disc panel */
    EVT_RADIOBOX(DiscType_Event, OpenDialog::OnDiscTypeChange)
    EVT_TEXT(DiscDevice_Event, OpenDialog::OnDiscPanelChange)
    EVT_TEXT(DiscTitle_Event, OpenDialog::OnDiscPanelChange)
    EVT_SPINCTRL(DiscTitle_Event, OpenDialog::OnDiscPanelChange)
    EVT_TEXT(DiscChapter_Event, OpenDialog::OnDiscPanelChange)
    EVT_SPINCTRL(DiscChapter_Event, OpenDialog::OnDiscPanelChange)

    /* Events generated by the net panel */
    EVT_RADIOBUTTON(NetRadio1_Event, OpenDialog::OnNetTypeChange)
    EVT_RADIOBUTTON(NetRadio2_Event, OpenDialog::OnNetTypeChange)
    EVT_RADIOBUTTON(NetRadio3_Event, OpenDialog::OnNetTypeChange)
    EVT_TEXT(NetPort1_Event, OpenDialog::OnNetPanelChange)
    EVT_SPINCTRL(NetPort1_Event, OpenDialog::OnNetPanelChange)
    EVT_TEXT(NetPort2_Event, OpenDialog::OnNetPanelChange)
    EVT_SPINCTRL(NetPort2_Event, OpenDialog::OnNetPanelChange)
    EVT_TEXT(NetPort3_Event, OpenDialog::OnNetPanelChange)
    EVT_SPINCTRL(NetPort3_Event, OpenDialog::OnNetPanelChange)
    EVT_TEXT(NetAddr2_Event, OpenDialog::OnNetPanelChange)
    EVT_TEXT(NetAddr3_Event, OpenDialog::OnNetPanelChange)

    /* Events generated by the subtitle file buttons */
    EVT_CHECKBOX(SubsFileEnable_Event, OpenDialog::OnSubsFileEnable)
    EVT_BUTTON(SubsFileSettings_Event, OpenDialog::OnSubsFileSettings)

    /* Events generated by the stream output buttons */
    EVT_CHECKBOX(SoutEnable_Event, OpenDialog::OnSoutEnable)
    EVT_BUTTON(SoutSettings_Event, OpenDialog::OnSoutSettings)

    /* Events generated by the demux dump buttons */
    EVT_CHECKBOX(DemuxDumpEnable_Event, OpenDialog::OnDemuxDumpEnable)
    EVT_TEXT(DemuxDump_Event, OpenDialog::OnDemuxDumpChange)
    EVT_BUTTON(DemuxDumpBrowse_Event, OpenDialog::OnDemuxDumpBrowse)

    /* Hide the window when the user closes the window */
    EVT_CLOSE(OpenDialog::OnCancel)

END_EVENT_TABLE()

/*****************************************************************************
 * Constructor.
 *****************************************************************************/
OpenDialog::OpenDialog( intf_thread_t *_p_intf, wxWindow *_p_parent,
                        int i_access_method, int i_arg ):
    wxFrame( _p_parent, -1, wxU(_("Open Target")), wxDefaultPosition,
             wxDefaultSize, wxDEFAULT_FRAME_STYLE )
{
    /* Initializations */
    p_intf = _p_intf;
    p_parent = _p_parent;
    SetIcon( *p_intf->p_sys->p_icon );
    file_dialog = NULL;
    sout_dialog = NULL;
    subsfile_dialog = NULL;
    demuxdump_dialog = NULL;

    /* Create a panel to put everything in */
    wxPanel *panel = new wxPanel( this, -1 );
    panel->SetAutoLayout( TRUE );

    /* Create MRL combobox */
    wxBoxSizer *mrl_sizer_sizer = new wxBoxSizer( wxHORIZONTAL );
    wxStaticBox *mrl_box = new wxStaticBox( panel, -1,
                               wxU(_("Media Resource Locator (MRL)")) );
    wxStaticBoxSizer *mrl_sizer = new wxStaticBoxSizer( mrl_box,
                                                        wxHORIZONTAL );
    wxStaticText *mrl_label = new wxStaticText( panel, -1,
                                                wxU(_("Open Target:")) );
    mrl_combo = new wxComboBox( panel, MRL_Event, wxT(""),
                                wxPoint(20,25), wxSize(120, -1),
                                0, NULL );
    mrl_combo->SetToolTip( wxU(_("You can use this field directly by typing "
        "the full MRL you want to open.\n""Alternatively, the field will be "
        "filled automatically when you use the controls below.")) );

    mrl_sizer->Add( mrl_label, 0, wxALL | wxALIGN_CENTER, 5 );
    mrl_sizer->Add( mrl_combo, 1, wxALL | wxALIGN_CENTER, 5 );
    mrl_sizer_sizer->Add( mrl_sizer, 1, wxEXPAND | wxALL, 5 );


    /* Create Static Text */
    wxStaticText *label = new wxStaticText( panel, -1,
        wxU(_("Alternatively, you can build an MRL using one of the "
              "following predefined targets:")) );

    /* Create Stream Output checkox */
    wxFlexGridSizer *sout_sizer = new wxFlexGridSizer( 2, 1, 20 );
    sout_checkbox = new wxCheckBox( panel, SoutEnable_Event,
                                    wxU(_("Stream output")) );
    sout_checkbox->SetToolTip( wxU(_("Use VLC as a stream server")) );
    sout_sizer->Add( sout_checkbox, 0,
                     wxALIGN_RIGHT | wxALIGN_CENTER_VERTICAL );
    sout_button = new wxButton( panel, SoutSettings_Event,
                                wxU(_("Settings...")) );
    sout_button->Disable();

    char *psz_sout = config_GetPsz( p_intf, "sout" );
    if( psz_sout && *psz_sout )
    {
        sout_checkbox->SetValue(TRUE);
        sout_button->Enable();
        subsfile_mrl.Add( wxString(wxT("sout=")) + wxU(psz_sout) );
    }
    if( psz_sout ) free( psz_sout );

    sout_sizer->Add( sout_button, 1, wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL );

    /* Create Demux Dump checkox */
    wxBoxSizer *demuxdump_sizer = new wxBoxSizer( wxHORIZONTAL );
    demuxdump_checkbox = new wxCheckBox( panel, DemuxDumpEnable_Event,
                               wxU(_("Capture input stream")) );
    demuxdump_checkbox->SetToolTip(
        wxU(_("Capture the stream you are playing to a file")) );
    demuxdump_textctrl = new wxTextCtrl( panel, DemuxDump_Event, wxT(""),
                                         wxDefaultPosition, wxDefaultSize,
                                         wxTE_PROCESS_ENTER );
    demuxdump_button = new wxButton( panel, DemuxDumpBrowse_Event,
                                     wxU(_("Browse...")) );

    char *psz_demuxdump = config_GetPsz( p_intf, "demuxdump-file" );
    if( psz_demuxdump && *psz_demuxdump )
    {
        demuxdump_textctrl->SetValue( wxU(psz_demuxdump) );
    }
    if( psz_demuxdump ) free( psz_demuxdump );

    demuxdump_textctrl->Disable();
    demuxdump_button->Disable();
    demuxdump_sizer->Add( demuxdump_checkbox, 0,
                          wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL );
    demuxdump_sizer->Add( demuxdump_button, 0,
                          wxLEFT | wxRIGHT | wxALIGN_CENTER_VERTICAL, 10 );
    demuxdump_sizer->Add( demuxdump_textctrl, 1, wxRIGHT, 10 );

    /* Separation */
    wxStaticLine *static_line = new wxStaticLine( panel, wxID_OK );

    /* Create the buttons */
    wxButton *ok_button = new wxButton( panel, wxID_OK, wxU(_("OK")) );
    ok_button->SetDefault();
    wxButton *cancel_button = new wxButton( panel, wxID_CANCEL,
                                            wxU(_("Cancel")) );

    /* Create notebook */
    notebook = new wxNotebook( panel, Notebook_Event );
    wxNotebookSizer *notebook_sizer = new wxNotebookSizer( notebook );

    notebook->AddPage( FilePanel( notebook ), wxU(_("File")),
                       i_access_method == FILE_ACCESS );
    notebook->AddPage( DiscPanel( notebook ), wxU(_("Disc")),
                       i_access_method == DISC_ACCESS );
    notebook->AddPage( NetPanel( notebook ), wxU(_("Network")),
                       i_access_method == NET_ACCESS );
#ifndef WIN32
    notebook->AddPage( SatPanel( notebook ), wxU(_("Satellite")),
                       i_access_method == SAT_ACCESS );
#endif

    /* Update Disc panel */
    wxCommandEvent dummy_event;
    OnDiscTypeChange( dummy_event );

    /* Update Net panel */
    dummy_event.SetId( NetRadio1_Event );
    OnNetTypeChange( dummy_event );

    /* Update MRL */
    wxNotebookEvent event( wxEVT_NULL, 0, i_access_method );
    OnPageChange( event );

    /* Place everything in sizers */
    wxBoxSizer *button_sizer = new wxBoxSizer( wxHORIZONTAL );
    button_sizer->Add( ok_button, 0, wxALL, 5 );
    button_sizer->Add( cancel_button, 0, wxALL, 5 );
    button_sizer->Layout();
    wxBoxSizer *main_sizer = new wxBoxSizer( wxVERTICAL );
    wxBoxSizer *panel_sizer = new wxBoxSizer( wxVERTICAL );
    panel_sizer->Add( mrl_sizer_sizer, 0, wxEXPAND, 5 );
    panel_sizer->Add( label, 0, wxEXPAND | wxALL, 5 );
    panel_sizer->Add( notebook_sizer, 1, wxEXPAND | wxALL, 5 );
    panel_sizer->Add( sout_sizer, 0, wxALIGN_LEFT | wxALL, 5 );
    panel_sizer->Add( demuxdump_sizer, 0, wxEXPAND | wxALIGN_LEFT | wxALL, 5 );
    panel_sizer->Add( static_line, 0, wxEXPAND | wxALL, 5 );
    panel_sizer->Add( button_sizer, 0, wxALIGN_LEFT | wxALL, 5 );
    panel_sizer->Layout();
    panel->SetSizerAndFit( panel_sizer );
    main_sizer->Add( panel, 1, wxGROW, 0 );
    main_sizer->Layout();
    SetSizerAndFit( main_sizer );
}

OpenDialog::~OpenDialog()
{
    /* Clean up */
    if( file_dialog ) delete file_dialog;
    if( sout_dialog ) delete sout_dialog;
    if( subsfile_dialog ) delete subsfile_dialog;
    if( demuxdump_dialog ) delete demuxdump_dialog;
}

int OpenDialog::Show( int i_access_method, int i_arg )
{
    int i_ret;
    notebook->SetSelection( i_access_method );
    i_ret = wxFrame::Show();
    Raise();
    SetFocus();
    return i_ret;
}

int OpenDialog::Show()
{
    int i_ret;
    i_ret = wxFrame::Show();
    Raise();
    SetFocus();
    return i_ret;
}

/*****************************************************************************
 * Private methods.
 *****************************************************************************/
wxPanel *OpenDialog::FilePanel( wxWindow* parent )
{
    wxPanel *panel = new wxPanel( parent, -1, wxDefaultPosition,
                                  wxSize(200, 200) );

    wxBoxSizer *sizer = new wxBoxSizer( wxVERTICAL );

    /* Create browse file line */
    wxBoxSizer *file_sizer = new wxBoxSizer( wxHORIZONTAL );

    file_combo = new wxComboBox( panel, FileName_Event, wxT(""),
                                 wxPoint(20,25), wxSize(200, -1), 0, NULL );
    wxButton *browse_button = new wxButton( panel, FileBrowse_Event,
                                            wxU(_("Browse...")) );
    file_sizer->Add( file_combo, 1, wxALL, 5 );
    file_sizer->Add( browse_button, 0, wxALL, 5 );

    /* Create Subtitles File checkox */
    wxFlexGridSizer *subsfile_sizer = new wxFlexGridSizer( 2, 1, 20 );
    subsfile_checkbox = new wxCheckBox( panel, SubsFileEnable_Event,
                                        wxU(_("Subtitles file")) );
    subsfile_checkbox->SetToolTip( wxU(_("Load an additional subtitles file. "
                                   "Currently only works with AVI files.")) );
    subsfile_sizer->Add( subsfile_checkbox, 0,
                         wxALIGN_RIGHT | wxALIGN_CENTER_VERTICAL );
    subsfile_button = new wxButton( panel, SubsFileSettings_Event,
                                    wxU(_("Settings...")) );
    subsfile_button->Disable();

    char *psz_subsfile = config_GetPsz( p_intf, "sub-file" );
    if( psz_subsfile && *psz_subsfile )
    {
        subsfile_checkbox->SetValue(TRUE);
        subsfile_button->Enable();
        subsfile_mrl.Add( wxString(wxT("sub-file=")) + wxU(psz_subsfile) );
    }
    if( psz_subsfile ) free( psz_subsfile );

    subsfile_sizer->Add( subsfile_button, 1,
                         wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL );

    sizer->Add( file_sizer, 0, wxEXPAND | wxALL, 5 );
    sizer->Add( subsfile_sizer, 0, wxEXPAND | wxALL, 5 );
    panel->SetSizerAndFit( sizer );
    return panel;
}

wxPanel *OpenDialog::DiscPanel( wxWindow* parent )
{
    wxPanel *panel = new wxPanel( parent, -1, wxDefaultPosition,
                                  wxSize(200, 200) );

    wxBoxSizer *sizer_row = new wxBoxSizer( wxVERTICAL );
    wxFlexGridSizer *sizer = new wxFlexGridSizer( 2, 3, 20 );

    static const wxString disc_type_array[] =
    {
        wxU(_("DVD (menus support)")),
        wxU(_("DVD")),
        wxU(_("VCD")),
        wxU(_("CD Audio"))

    };

    disc_type = new wxRadioBox( panel, DiscType_Event, wxU(_("Disc type")),
                                wxDefaultPosition, wxDefaultSize,
                                WXSIZEOF(disc_type_array), disc_type_array,
                                WXSIZEOF(disc_type_array), wxRA_SPECIFY_COLS );
    sizer_row->Add( disc_type, 0, wxEXPAND | wxALL, 5 );

    wxStaticText *label = new wxStaticText( panel, -1, wxU(_("Device name")) );
    disc_device = new wxTextCtrl( panel, DiscDevice_Event, wxT(""),
                                  wxDefaultPosition, wxDefaultSize,
                                  wxTE_PROCESS_ENTER);

    sizer->Add( label, 0, wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL );
    sizer->Add( disc_device, 1, wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL );

    label = new wxStaticText( panel, -1, wxU(_("Title")) );
    disc_title = new wxSpinCtrl( panel, DiscTitle_Event );

    sizer->Add( label, 0, wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL );
    sizer->Add( disc_title, 1, wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL );

    label = new wxStaticText( panel, -1, wxU(_("Chapter")) );
    disc_chapter = new wxSpinCtrl( panel, DiscChapter_Event );
    sizer->Add( label, 0, wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL );
    sizer->Add( disc_chapter, 1, wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL );
    sizer_row->Add( sizer, 0, wxEXPAND | wxALL, 5 );

    panel->SetSizerAndFit( sizer_row );
    return panel;
}

wxPanel *OpenDialog::NetPanel( wxWindow* parent )
{
    int i;
    wxPanel *panel = new wxPanel( parent, -1, wxDefaultPosition,
                                  wxSize(200, 200) );

    wxBoxSizer *sizer_row = new wxBoxSizer( wxVERTICAL );
    wxFlexGridSizer *sizer = new wxFlexGridSizer( 2, 3, 20 );

    static const wxString net_type_array[] =
    {
        wxU(_("UDP/RTP")),
        wxU(_("UDP/RTP Multicast")),
        wxU(_("HTTP/FTP/MMS"))
    };

    for( i=0; i<3; i++ )
    {
        net_radios[i] = new wxRadioButton( panel, NetRadio1_Event + i,
                                           net_type_array[i],
                                           wxDefaultPosition, wxDefaultSize,
                                           wxRB_SINGLE );

        net_subpanels[i] = new wxPanel( panel, -1,
                                        wxDefaultPosition, wxDefaultSize );
    }

    /* UDP/RTP row */
    wxFlexGridSizer *subpanel_sizer;
    wxStaticText *label;
    i_net_ports[0] = config_GetInt( p_intf, "server-port" );
    subpanel_sizer = new wxFlexGridSizer( 2, 1, 20 );
    label = new wxStaticText( net_subpanels[0], -1, wxU(_("Port")) );
    net_ports[0] = new wxSpinCtrl( net_subpanels[0], NetPort1_Event,
                                   wxString::Format(wxT("%d"), i_net_ports[0]),
                                   wxDefaultPosition, wxDefaultSize,
                                   wxSP_ARROW_KEYS,
                                   0, 16000, i_net_ports[0] );

    subpanel_sizer->Add( label, 0, wxALIGN_RIGHT | wxALIGN_CENTER_VERTICAL );
    subpanel_sizer->Add( net_ports[0], 1,
                         wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL );
    net_subpanels[0]->SetSizerAndFit( subpanel_sizer );
    net_radios[0]->SetValue( TRUE );

    /* UDP/RTP Multicast row */
    subpanel_sizer = new wxFlexGridSizer( 4, 1, 20 );
    label = new wxStaticText( net_subpanels[1], -1, wxU(_("Address")) );
    net_addrs[1] = new wxTextCtrl( net_subpanels[1], NetAddr2_Event, wxT(""),
                                   wxDefaultPosition, wxDefaultSize,
                                   wxTE_PROCESS_ENTER);
    subpanel_sizer->Add( label, 0, wxALIGN_RIGHT | wxALIGN_CENTER_VERTICAL );
    subpanel_sizer->Add( net_addrs[1], 1,
                         wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL );

    label = new wxStaticText( net_subpanels[1], -1, wxU(_("Port")) );
    i_net_ports[1] = i_net_ports[0];
    net_ports[1] = new wxSpinCtrl( net_subpanels[1], NetPort2_Event,
                                   wxString::Format(wxT("%d"), i_net_ports[1]),
                                   wxDefaultPosition, wxDefaultSize,
                                   wxSP_ARROW_KEYS,
                                   0, 16000, i_net_ports[1] );

    subpanel_sizer->Add( label, 0, wxALIGN_RIGHT | wxALIGN_CENTER_VERTICAL );
    subpanel_sizer->Add( net_ports[1], 1,
                         wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL );
    net_subpanels[1]->SetSizerAndFit( subpanel_sizer );

    /* HTTP row */
    subpanel_sizer = new wxFlexGridSizer( 2, 1, 20 );
    label = new wxStaticText( net_subpanels[2], -1, wxU(_("URL")) );
    net_addrs[2] = new wxTextCtrl( net_subpanels[2], NetAddr3_Event, wxT(""),
                                   wxDefaultPosition, wxSize( 200, -1 ),
                                   wxTE_PROCESS_ENTER);
    subpanel_sizer->Add( label, 0, wxALIGN_RIGHT | wxALIGN_CENTER_VERTICAL );
    subpanel_sizer->Add( net_addrs[2], 1,
                         wxEXPAND | wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL );
    net_subpanels[2]->SetSizerAndFit( subpanel_sizer );

    /* Stuff everything into the main panel */
    for( i=0; i<3; i++ )
    {
        sizer->Add( net_radios[i], 0, wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL |
                    wxALL, 5 );
        sizer->Add( net_subpanels[i], 1, wxEXPAND | wxALIGN_LEFT |
                    wxALIGN_CENTER_VERTICAL | wxALL, 5  );
    }

    sizer_row->Add( sizer, 0, wxEXPAND | wxALL, 5 );

    panel->SetSizerAndFit( sizer_row );
    return panel;
}

wxPanel *OpenDialog::SatPanel( wxWindow* parent )
{
    wxPanel *panel = new wxPanel( parent, -1, wxDefaultPosition,
                                  wxSize(200, 200) );
    return panel;
}

void OpenDialog::UpdateMRL( int i_access_method )
{
    wxString demux, mrltemp;

    i_current_access_method = i_access_method;

    /* Check if the user asked for demuxdump */
    if( demuxdump_checkbox->GetValue() )
    {
        demux = wxT("/demuxdump");
    }

    switch( i_access_method )
    {
    case FILE_ACCESS:
        //mrltemp = wxT("file") + demux + wxT(":") + file_combo->GetValue();
        mrltemp = file_combo->GetValue();
        break;
    case DISC_ACCESS:
        mrltemp = ( disc_type->GetSelection() == 0 ? wxT("dvd") :
                disc_type->GetSelection() == 1 ? wxT("dvdsimple") :
                disc_type->GetSelection() == 2 ? wxT("vcd") : wxT("cdda") )
                  + demux + wxT(":")
                  + disc_device->GetLineText(0)
                  + wxString::Format( wxT("@%d:%d"),
                                      i_disc_title, i_disc_chapter );
        break;
    case NET_ACCESS:
        switch( i_net_type )
        {
        case 0:
            if( i_net_ports[0] !=
                config_GetInt( p_intf, "server-port" ) )
            {
                mrltemp = wxT("udp") + demux +
                          wxString::Format( wxT("://@:%d"),
                                            i_net_ports[0] );
            }
            else
            {
                mrltemp = wxT("udp") + demux + wxT("://");
            }
            break;

        case 1:
            mrltemp = wxT("udp") + demux + wxT("://@") +
                      net_addrs[1]->GetLineText(0);
            if( i_net_ports[1] !=
                config_GetInt( p_intf, "server-port" ) )
            {
                mrltemp = mrltemp + wxString::Format( wxT(":%d"),
                                              i_net_ports[1] );
            }
            break;

        case 2:
            /* http access */     
            mrltemp = wxT("http") + demux + wxT("://") +
                      net_addrs[2]->GetLineText(0);
            break;
        }
        break;
    case SAT_ACCESS:
        mrltemp = wxT("satellite") + demux + wxT("://");
        break;
    default:
        break;
    }

    mrl_combo->SetValue( mrltemp );
}

/*****************************************************************************
 * Events methods.
 *****************************************************************************/
void OpenDialog::OnOk( wxCommandEvent& WXUNUSED(event) )
{
    mrl = SeparateEntries( mrl_combo->GetValue() );
    mrl_combo->Append( mrl_combo->GetValue() );
    if( mrl_combo->GetCount() > 10 ) mrl_combo->Delete( 0 );
    mrl_combo->SetSelection( mrl_combo->GetCount() - 1 );

    /* Update the playlist */
    playlist_t *p_playlist =
        (playlist_t *)vlc_object_find( p_intf, VLC_OBJECT_PLAYLIST,
				       FIND_ANYWHERE );
    if( p_playlist == NULL ) return;

    for( int i = 0; i < (int)mrl.GetCount(); i++ )
    {
        int i_options = 0, i_total_options;
        char **ppsz_options = NULL;

        /* Count the input options */
        while( i + i_options + 1 < (int)mrl.GetCount() &&
               ((const char *)mrl[i + i_options + 1].mb_str())[0] == ':' )
        {
            i_options++;
        }

        /* Allocate ppsz_options */
        for( int j = 0; j < i_options; j++ )
        {
            if( !ppsz_options )
                ppsz_options = (char **)malloc( sizeof(char *) * i_options );

            ppsz_options[j] = strdup( mrl[i + j  + 1].mb_str() );
        }

	i_total_options = i_options;

        /* Get the options from the subtitles dialog */
        if( subsfile_checkbox->IsChecked() && subsfile_mrl.GetCount() )
        {
            ppsz_options = (char **)realloc( ppsz_options, sizeof(char *) *
                               (i_total_options + subsfile_mrl.GetCount()) );

            for( int j = 0; j < (int)subsfile_mrl.GetCount(); j++ )
            {
                ppsz_options[i_total_options + j] =
                    strdup( subsfile_mrl[j].mb_str() );
            }

            i_total_options += subsfile_mrl.GetCount();
        }

        /* Get the options from the stream output dialog */
        if( sout_checkbox->IsChecked() && sout_mrl.GetCount() )
        {
            ppsz_options = (char **)realloc( ppsz_options, sizeof(char *) *
                               (i_total_options + sout_mrl.GetCount()) );

            for( int j = 0; j < (int)sout_mrl.GetCount(); j++ )
            {
                ppsz_options[i_total_options + j] =
                    strdup( sout_mrl[j].mb_str() );
            }

            i_total_options += sout_mrl.GetCount();
        }

        playlist_Add( p_playlist, (const char *)mrl[i].mb_str(),
                      (const char **)ppsz_options, i_total_options,
                      PLAYLIST_APPEND | (i ? 0 : PLAYLIST_GO), PLAYLIST_END );

        /* clean up */
        for( int j = 0; j < i_total_options; j++ )
            free( ppsz_options[j] );
        if( ppsz_options ) free( ppsz_options );

        i += i_options;
   }

    //TogglePlayButton( PLAYING_S );

    vlc_object_release( p_playlist );

    Hide();
}

void OpenDialog::OnCancel( wxCommandEvent& WXUNUSED(event) )
{
    Hide();
}

void OpenDialog::OnPageChange( wxNotebookEvent& event )
{
    UpdateMRL( event.GetSelection() );
}

void OpenDialog::OnMRLChange( wxCommandEvent& event )
{
    //mrl = SeparateEntries( event.GetString() );
}

/*****************************************************************************
 * File panel event methods.
 *****************************************************************************/
void OpenDialog::OnFilePanelChange( wxCommandEvent& WXUNUSED(event) )
{
    UpdateMRL( FILE_ACCESS );
}

void OpenDialog::OnFileBrowse( wxCommandEvent& WXUNUSED(event) )
{
    if( file_dialog == NULL )
        file_dialog = new wxFileDialog( this, wxU(_("Open file")),
            wxT(""), wxT(""), wxT("*"), wxOPEN | wxMULTIPLE );

    if( file_dialog && file_dialog->ShowModal() == wxID_OK )
    {
        wxArrayString paths;
        wxString path;

        file_dialog->GetPaths( paths );

        for( size_t i = 0; i < paths.GetCount(); i++ )
        {
            if( paths[i].Find( wxT(' ') ) >= 0 )
                path += wxT("\"") + paths[i] + wxT("\" ");
            else
                path += paths[i] + wxT(" ");
        }

        file_combo->SetValue( path );
        file_combo->Append( path );
        if( file_combo->GetCount() > 10 ) file_combo->Delete( 0 );
        UpdateMRL( FILE_ACCESS );
    }
}

/*****************************************************************************
 * Disc panel event methods.
 *****************************************************************************/
void OpenDialog::OnDiscPanelChange( wxCommandEvent& event )
{
    if( event.GetId() == DiscTitle_Event )
        i_disc_title = event.GetInt();
    else if( event.GetId() == DiscChapter_Event )
        i_disc_chapter = event.GetInt();

    UpdateMRL( DISC_ACCESS );
}

void OpenDialog::OnDiscTypeChange( wxCommandEvent& WXUNUSED(event) )
{
    char *psz_device;

    switch( disc_type->GetSelection() )
    {
    case 2:
        psz_device = config_GetPsz( p_intf, "vcd" );
        disc_device->SetValue( psz_device ? wxU(psz_device) : wxT("") );
        break;

    default:
        psz_device = config_GetPsz( p_intf, "dvd" );
        disc_device->SetValue( psz_device ? wxU(psz_device) : wxT("") );
        break;
    }

    if( psz_device ) free( psz_device );

    switch( disc_type->GetSelection() )
    {
    case 0:
        disc_title->SetRange( 0, 255 );
        disc_title->SetValue( 0 );
        i_disc_title = 0;
        break;

    default:
        disc_title->SetRange( 1, 255 );
        disc_title->SetValue( 1 );
        i_disc_title = 1;
        break;
    }

    i_disc_chapter = 1;
    disc_chapter->SetRange( 1, 255 );
    disc_chapter->SetValue( 1 );

    UpdateMRL( DISC_ACCESS );
}

/*****************************************************************************
 * Net panel event methods.
 *****************************************************************************/
void OpenDialog::OnNetPanelChange( wxCommandEvent& event )
{
    if( event.GetId() >= NetPort1_Event && event.GetId() <= NetPort3_Event )
    {
        i_net_ports[event.GetId() - NetPort1_Event] = event.GetInt();
    }

    UpdateMRL( NET_ACCESS );
}

void OpenDialog::OnNetTypeChange( wxCommandEvent& event )
{
    int i;

    i_net_type = event.GetId() - NetRadio1_Event;

    for(i=0; i<3; i++)
    {
        net_radios[i]->SetValue( event.GetId() == (NetRadio1_Event+i) );
        net_subpanels[i]->Enable( event.GetId() == (NetRadio1_Event+i) );
    }

    UpdateMRL( NET_ACCESS );
}

/*****************************************************************************
 * Subtitles file event methods.
 *****************************************************************************/
void OpenDialog::OnSubsFileEnable( wxCommandEvent& event )
{
    subsfile_button->Enable( event.GetInt() != 0 );
    if( event.GetInt() && demuxdump_checkbox->IsChecked() )
    {
        demuxdump_checkbox->SetValue( 0 );
        wxCommandEvent event = wxCommandEvent( wxEVT_NULL );
        event.SetInt( 0 );
        OnDemuxDumpEnable( event );
    }
}

void OpenDialog::OnSubsFileSettings( wxCommandEvent& WXUNUSED(event) )
{
    /* Show/hide the open dialog */
    if( subsfile_dialog == NULL )
        subsfile_dialog = new SubsFileDialog( p_intf, this );

    if( subsfile_dialog && subsfile_dialog->ShowModal() == wxID_OK )
    {
        subsfile_mrl.Empty();
        subsfile_mrl.Add( wxString(wxT("sub-file=")) +
                          subsfile_dialog->file_combo->GetValue() );
        subsfile_mrl.Add( wxString::Format( wxT("sub-delay=%i"),
                          subsfile_dialog->delay_spinctrl->GetValue() ) );
        subsfile_mrl.Add( wxString::Format( wxT("sub-fps=%i"),
                          subsfile_dialog->fps_spinctrl->GetValue() ) );
    }
}

/*****************************************************************************
 * Stream output event methods.
 *****************************************************************************/
void OpenDialog::OnSoutEnable( wxCommandEvent& event )
{
    sout_button->Enable( event.GetInt() != 0 );
    if( event.GetInt() && demuxdump_checkbox->IsChecked() )
    {
        demuxdump_checkbox->SetValue( 0 );
        wxCommandEvent event = wxCommandEvent( wxEVT_NULL );
        event.SetInt( 0 );
        OnDemuxDumpEnable( event );
    }
}

void OpenDialog::OnSoutSettings( wxCommandEvent& WXUNUSED(event) )
{
    /* Show/hide the open dialog */
    if( sout_dialog == NULL )
        sout_dialog = new SoutDialog( p_intf, this );

    if( sout_dialog && sout_dialog->ShowModal() == wxID_OK )
    {
        sout_mrl = sout_dialog->GetOptions();
    }
}

/*****************************************************************************
 * Demux dump event methods.
 *****************************************************************************/
void OpenDialog::OnDemuxDumpEnable( wxCommandEvent& event )
{
    demuxdump_textctrl->Enable( event.GetInt() != 0 );
    demuxdump_button->Enable( event.GetInt() != 0 );

    if( event.GetInt() )
    {
        sout_checkbox->SetValue( 0 );
        subsfile_checkbox->SetValue( 0 );
        wxCommandEvent event = wxCommandEvent( wxEVT_NULL );
        event.SetInt( 0 );
        OnSoutEnable( event );
        OnSubsFileEnable( event );
    }

    UpdateMRL( i_current_access_method );
}

void OpenDialog::OnDemuxDumpBrowse( wxCommandEvent& WXUNUSED(event) )
{
    if( demuxdump_dialog == NULL )
        demuxdump_dialog = new wxFileDialog( this, wxU(_("Save file")),
                               wxT(""), wxT(""), wxT("*"), wxSAVE );

    if( demuxdump_dialog && demuxdump_dialog->ShowModal() == wxID_OK )
    {
        demuxdump_textctrl->SetValue( demuxdump_dialog->GetPath() );
        wxCommandEvent event = wxCommandEvent( wxEVT_NULL );
        OnDemuxDumpChange( event );
    }
}

void OpenDialog::OnDemuxDumpChange( wxCommandEvent& WXUNUSED(event) )
{
    config_PutPsz( p_intf, "demuxdump-file",
                   demuxdump_textctrl->GetValue().mb_str() );
}

/*****************************************************************************
 * Utility functions.
 *****************************************************************************/
wxArrayString SeparateEntries( wxString entries )
{
    vlc_bool_t b_quotes_mode = VLC_FALSE;

    wxArrayString entries_array;
    wxString entry;

    wxStringTokenizer token( entries, wxT(" \t\r\n\""), wxTOKEN_RET_DELIMS );

    while( token.HasMoreTokens() )
    {
        entry += token.GetNextToken();

        if( entry.IsEmpty() ) continue;

        if( !b_quotes_mode && entry.Last() == wxT('\"') )
        {
            /* Enters quotes mode */
            entry.RemoveLast();
            b_quotes_mode = VLC_TRUE;
        }
        else if( b_quotes_mode && entry.Last() == wxT('\"') )
        {
            /* Finished the quotes mode */
            entry.RemoveLast();
            if( !entry.IsEmpty() ) entries_array.Add( entry );
            entry.Empty();
            b_quotes_mode = VLC_FALSE;
        }
        else if( !b_quotes_mode && entry.Last() != wxT('\"') )
        {
            /* we found a non-quoted standalone string */
            if( token.HasMoreTokens() ||
                entry.Last() == wxT(' ') || entry.Last() == wxT('\t') ||
                entry.Last() == wxT('\r') || entry.Last() == wxT('\n') )
                entry.RemoveLast();
            if( !entry.IsEmpty() ) entries_array.Add( entry );
            entry.Empty();
        }
        else
        {;}
    }

    if( !entry.IsEmpty() ) entries_array.Add( entry );

    return entries_array;
}
